<?php

declare(strict_types=1);

/**
 * Licensed under the MIT license.
 *
 * For the full copyright and license information, please view the LICENSE file.
 *
 * @author Rémi Lanvin <remi@cloudconnected.fr>
 *
 * @see https://github.com/rlanvin/php-ip
 */

namespace PhpIP\Tests;

use PhpIP\IPv4Block;
use PHPUnit\Framework\TestCase;

class IPv4BlockTest extends TestCase
{
    // see http://www.miniwebtool.com/ip-address-to-binary-converter/
    // and http://www.miniwebtool.com/ip-address-to-hex-converter
    public function validBlocks()
    {
        return [
            //CIDR            Mask               Delta              First IP          Last IP
            ['0.0.0.0/0',    '0.0.0.0',         '255.255.255.255', '0.0.0.0',        '255.255.255.255'],
            ['130.0.0.0/1',  '128.0.0.0',       '127.255.255.255', '128.0.0.0',      '255.255.255.255'],
            ['130.0.0.0/2',  '192.0.0.0',       '63.255.255.255',  '128.0.0.0',      '191.255.255.255'],
            ['130.0.0.0/3',  '224.0.0.0',       '31.255.255.255',  '128.0.0.0',      '159.255.255.255'],
            ['130.0.0.0/4',  '240.0.0.0',       '15.255.255.255',  '128.0.0.0',      '143.255.255.255'],
            ['130.0.0.0/5',  '248.0.0.0',       '7.255.255.255',   '128.0.0.0',      '135.255.255.255'],
            ['130.0.0.0/6',  '252.0.0.0',       '3.255.255.255',   '128.0.0.0',      '131.255.255.255'],
            ['128.0.1.0/7',  '254.0.0.0',       '1.255.255.255',   '128.0.0.0',      '129.255.255.255'],
            ['127.0.0.0/8',  '255.0.0.0',       '0.255.255.255',   '127.0.0.0',      '127.255.255.255'],
            ['127.0.0.0/9',  '255.128.0.0',     '0.127.255.255',   '127.0.0.0',      '127.127.255.255'],
            ['127.0.0.0/10', '255.192.0.0',     '0.63.255.255',    '127.0.0.0',      '127.63.255.255'],
            ['127.0.0.0/11', '255.224.0.0',     '0.31.255.255',    '127.0.0.0',      '127.31.255.255'],
            ['127.0.0.0/12', '255.240.0.0',     '0.15.255.255',    '127.0.0.0',      '127.15.255.255'],
            ['127.0.0.0/13', '255.248.0.0',     '0.7.255.255',     '127.0.0.0',      '127.7.255.255'],
            ['127.0.0.0/14', '255.252.0.0',     '0.3.255.255',     '127.0.0.0',      '127.3.255.255'],
            ['127.0.0.0/15', '255.254.0.0',     '0.1.255.255',     '127.0.0.0',      '127.1.255.255'],
            ['127.0.0.1/16', '255.255.0.0',     '0.0.255.255',     '127.0.0.0',      '127.0.255.255'],
            ['127.0.0.1/17', '255.255.128.0',   '0.0.127.255',     '127.0.0.0',      '127.0.127.255'],
            ['127.0.0.1/18', '255.255.192.0',   '0.0.63.255',      '127.0.0.0',      '127.0.63.255'],
            ['127.0.0.1/19', '255.255.224.0',   '0.0.31.255',      '127.0.0.0',      '127.0.31.255'],
            ['127.0.0.1/20', '255.255.240.0',   '0.0.15.255',      '127.0.0.0',      '127.0.15.255'],
            ['127.0.0.1/21', '255.255.248.0',   '0.0.7.255',       '127.0.0.0',      '127.0.7.255'],
            ['127.0.0.1/22', '255.255.252.0',   '0.0.3.255',       '127.0.0.0',      '127.0.3.255'],
            ['127.0.0.1/23', '255.255.254.0',   '0.0.1.255',       '127.0.0.0',      '127.0.1.255'],
            ['127.0.0.1/24', '255.255.255.0',   '0.0.0.255',       '127.0.0.0',      '127.0.0.255'],
            ['127.0.0.1/25', '255.255.255.128', '0.0.0.127',       '127.0.0.0',      '127.0.0.127'],
            ['127.0.0.1/26', '255.255.255.192', '0.0.0.63',        '127.0.0.0',      '127.0.0.63'],
            ['127.0.0.1/27', '255.255.255.224', '0.0.0.31',        '127.0.0.0',      '127.0.0.31'],
            ['127.0.0.1/28', '255.255.255.240', '0.0.0.15',        '127.0.0.0',      '127.0.0.15'],
            ['127.0.0.1/29', '255.255.255.248', '0.0.0.7',         '127.0.0.0',      '127.0.0.7'],
            ['127.0.0.1/30', '255.255.255.252', '0.0.0.3',         '127.0.0.0',      '127.0.0.3'],
            ['127.0.0.1/31', '255.255.255.254', '0.0.0.1',         '127.0.0.0',      '127.0.0.1'],
            ['127.0.0.1/32', '255.255.255.255', '0.0.0.0',         '127.0.0.1',      '127.0.0.1'],

            //Net+Netmask                 Mask               Delta              First IP          Last IP
            ['0.0.0.0/0.0.0.0',           '0.0.0.0',         '255.255.255.255', '0.0.0.0',        '255.255.255.255'],
            ['130.0.0.0/128.0.0.0',       '128.0.0.0',       '127.255.255.255', '128.0.0.0',      '255.255.255.255'],
            ['130.0.0.0/192.0.0.0',       '192.0.0.0',       '63.255.255.255',  '128.0.0.0',      '191.255.255.255'],
            ['130.0.0.0/224.0.0.0',       '224.0.0.0',       '31.255.255.255',  '128.0.0.0',      '159.255.255.255'],
            ['130.0.0.0/240.0.0.0',       '240.0.0.0',       '15.255.255.255',  '128.0.0.0',      '143.255.255.255'],
            ['130.0.0.0/248.0.0.0',       '248.0.0.0',       '7.255.255.255',   '128.0.0.0',      '135.255.255.255'],
            ['130.0.0.0/252.0.0.0',       '252.0.0.0',       '3.255.255.255',   '128.0.0.0',      '131.255.255.255'],
            ['128.0.1.0/254.0.0.0',       '254.0.0.0',       '1.255.255.255',   '128.0.0.0',      '129.255.255.255'],
            ['127.0.0.0/255.0.0.0',       '255.0.0.0',       '0.255.255.255',   '127.0.0.0',      '127.255.255.255'],
            ['127.0.0.0/255.128.0.0',     '255.128.0.0',     '0.127.255.255',   '127.0.0.0',      '127.127.255.255'],
            ['127.0.0.0/255.192.0.0',     '255.192.0.0',     '0.63.255.255',    '127.0.0.0',      '127.63.255.255'],
            ['127.0.0.0/255.224.0.0',     '255.224.0.0',     '0.31.255.255',    '127.0.0.0',      '127.31.255.255'],
            ['127.0.0.0/255.240.0.0',     '255.240.0.0',     '0.15.255.255',    '127.0.0.0',      '127.15.255.255'],
            ['127.0.0.0/255.248.0.0',     '255.248.0.0',     '0.7.255.255',     '127.0.0.0',      '127.7.255.255'],
            ['127.0.0.0/255.252.0.0',     '255.252.0.0',     '0.3.255.255',     '127.0.0.0',      '127.3.255.255'],
            ['127.0.0.0/255.254.0.0',     '255.254.0.0',     '0.1.255.255',     '127.0.0.0',      '127.1.255.255'],
            ['127.0.0.1/255.255.0.0',     '255.255.0.0',     '0.0.255.255',     '127.0.0.0',      '127.0.255.255'],
            ['127.0.0.1/255.255.128.0',   '255.255.128.0',   '0.0.127.255',     '127.0.0.0',      '127.0.127.255'],
            ['127.0.0.1/255.255.192.0',   '255.255.192.0',   '0.0.63.255',      '127.0.0.0',      '127.0.63.255'],
            ['127.0.0.1/255.255.224.0',   '255.255.224.0',   '0.0.31.255',      '127.0.0.0',      '127.0.31.255'],
            ['127.0.0.1/255.255.240.0',   '255.255.240.0',   '0.0.15.255',      '127.0.0.0',      '127.0.15.255'],
            ['127.0.0.1/255.255.248.0',   '255.255.248.0',   '0.0.7.255',       '127.0.0.0',      '127.0.7.255'],
            ['127.0.0.1/255.255.252.0',   '255.255.252.0',   '0.0.3.255',       '127.0.0.0',      '127.0.3.255'],
            ['127.0.0.1/255.255.254.0',   '255.255.254.0',   '0.0.1.255',       '127.0.0.0',      '127.0.1.255'],
            ['127.0.0.1/255.255.255.0',   '255.255.255.0',   '0.0.0.255',       '127.0.0.0',      '127.0.0.255'],
            ['127.0.0.1/255.255.255.128', '255.255.255.128', '0.0.0.127',       '127.0.0.0',      '127.0.0.127'],
            ['127.0.0.1/255.255.255.192', '255.255.255.192', '0.0.0.63',        '127.0.0.0',      '127.0.0.63'],
            ['127.0.0.1/255.255.255.224', '255.255.255.224', '0.0.0.31',        '127.0.0.0',      '127.0.0.31'],
            ['127.0.0.1/255.255.255.240', '255.255.255.240', '0.0.0.15',        '127.0.0.0',      '127.0.0.15'],
            ['127.0.0.1/255.255.255.248', '255.255.255.248', '0.0.0.7',         '127.0.0.0',      '127.0.0.7'],
            ['127.0.0.1/255.255.255.252', '255.255.255.252', '0.0.0.3',         '127.0.0.0',      '127.0.0.3'],
            ['127.0.0.1/255.255.255.254', '255.255.255.254', '0.0.0.1',         '127.0.0.0',      '127.0.0.1'],
            ['127.0.0.1/255.255.255.255', '255.255.255.255', '0.0.0.0',         '127.0.0.1',      '127.0.0.1'],
        ];
    }

    /**
     * @dataProvider validBlocks
     */
    public function testConstructValid($block, $mask, $delta, $first_ip, $last_ip)
    {
        $instance = new IPv4Block($block);
        $this->assertEquals($mask, (string) $instance->getMask(), "Mask of $block");
        $this->assertEquals($delta, (string) $instance->getDelta(), "Delta of $block");
        $this->assertEquals($first_ip, (string) $instance->getFirstIp(), "First IP of $block");
        $this->assertEquals($last_ip, (string) $instance->getLastIp(), "Last IP of $block");
    }

    public function invalidBlocks()
    {
        return [
            ['127.0.2666.1/24'],
            ['127.0.0.1/45'],
            ['127.0.0.1/255.255.2555.0'],
            ["\t"],
            ['abc'],
            [12.3],
            [-12.3],
            ['-1'],
            ['4294967296'],
            ['2a01:8200::'],
            ['::1'],
            ['::1/128'],
            ['2a01:8200::/32'],
        ];
    }

    /**
     * @dataProvider invalidBlocks
     */
    public function testConstructInvalid($block)
    {
        $this->expectException(\InvalidArgumentException::class);

        $instance = new IPv4Block($block);
    }

    public function testIterator()
    {
        $expectation = [
            '192.168.0.0',
            '192.168.0.1',
            '192.168.0.2',
            '192.168.0.3',
            '192.168.0.4',
            '192.168.0.5',
            '192.168.0.6',
            '192.168.0.7',
            '192.168.0.8',
            '192.168.0.9',
            '192.168.0.10',
            '192.168.0.11',
            '192.168.0.12',
            '192.168.0.13',
            '192.168.0.14',
            '192.168.0.15',
        ];

        $subnet = new IPv4Block('192.168.0.0/28');

        $this->assertEquals($expectation, iterator_to_array($subnet->getIterator()));
    }
}
