# 1.x to 2.x migration guide

The release of the `2.0` version of the library introduced many breaking changes in the API.

This document is provided to support migration from a `1.x` to a `2.x` version of the library.

This guide covers the three components exposed by the library's API:
 * `VASTClient`
 * `VASTParser`
 * `VASTTracker`

## No more static methods and properties

Vast Client JS `1.x` made strong use of static methods and properties. The `2.0` version removes all the static components in favor of a non-static access through an instance of the class.

`1.x` - Direct access to static components of the classes (without getting any instance of them):
```Javascript
import DMVAST from 'vast-client'

const vastUrl = 'http://example.dailymotion.com/vast.xml';

DMVAST.client.get(VASTUrl, cb);
DMVAST.parser.parse(VASTUrl, options, function(response, error) {
  // process the VAST response
});

// Only the tracker component had already to be accessed through an instance of the class
var vastTracker = new DMVAST.tracker(ad, creative);
```

`2.x` - To access any property or method of a class you first need to get an instance of the class using the constructor:
```Javascript
import {
  VASTClient,
  VASTParser,
  VASTTracker
} from 'vast-client'

const vastUrl = 'http://example.dailymotion.com/vast.xml';

const client = new VASTClient();
const parser = new VASTParser();
const tracker = new VASTTracker(client, ad, creative);

// This would cause an error since methods are not static anymore
VASTClient.get(vastUrl, cb);
// Call get from the instance
client.get(vastUrl, cb);
```

Another major impact of this change is that an action made on a component will now affect only the istance of that component on which the action has been made.

`1.x` - Adding an url template filter to the `VASTParser` class will affect the static properties of the class. Meaning that any class using the `VASTParser` will use this url template filter now:
```Javascript
import DMVAST from 'vast-client'

DMVAST.parser.addURLTemplateFilter( function(vastUrl) {
    return url.replace('[DOMAIN]', 'mywebsite.com');
});

// Calling the get method of the client (which uses DMVAST.parser internally)
// will be affected by the above call to DMVAST.parser.addURLTemplateFilter
DMVAST.client.get(vastUrl, options, cb);
```

`2.x` - Adding an url template filter to an instance of `VASTParser` will affect only the behavior of that instance:
```Javascript
import {
  VASTClient,
  VASTParser
} from 'vast-client'

const client = new VASTClient();
const parser = new VASTParser();

parser.addURLTemplateFilter( vastUrl => url.replace('[DOMAIN]', 'mywebsite.com') );

// Calling the get method of the client (which uses an instance of VASTParser internally)
// will not be affected by the above call to parser.addURLTemplateFilter
// since the internal instance is a different one
client.get(vastUrl, options, cb);

// To have an effect on the client get behavior you need to call
// addURLTemplateFilter from the istance of VASTParser used by the client
client.vastParser.addURLTemplateFilter( vastUrl => url.replace('[DOMAIN]', 'mywebsite.com') );

// This call will now use the new template filter
client.get(vastUrl, options, cb);
```

## Promises over callback
Every callback function has been replaced with an es6 `Promise`.
```Javascript
// before
vastClient.get('https://www.examplevast.com/vast.xml', (err, res) => {
  if (err) {
    // Deal with the error
  }

  // Deal with the result
}

// after
vastClient.get('https://www.examplevast.com/vast.xml')
  .then(res => {
    // Do something with the parsed VAST response
  })
  .catch(err => {
    // Deal with the error
  })
});
```

## Changes in Client API

### Constructor and properties
As mentioned in the first section, now you can use the client only through an instance. Documentation for the constructor is available in the `VASTClient` API docs.

The constructor is now the preferred way to set the class properties. Direct access to properties should be avoided when possible.

### Breaking changes in method signatures
 * `get(url, [options,] cb)` becomes `get(url, options)`

## Changes in Parser API

### Constructor and properties
As mentioned in the first section, you can now only use the parser through an instance. Documentation for the constructor is available in the `VASTParser` API docs.

### Breaking changes in methods signatures
These changes are mostly due to the semantic meaning of methods names.

 * `parse(url, [options,] cb)` becomes `getAndParseVAST(url, options)`
 * `load(vastXml, [options,] cb)` becomes `parseVAST(vastXml, options)`

`1.x` - Fetch and parse a VAST or simply parse a VAST:
```Javascript
// To fetch and parse a VAST
var vastUrl = 'http://example.dailymotion.com/vast.xml';

DMVAST.parser.parse(vastUrl, options, cb);

// To parse an already fetched VAST xml
var vastXml = (new window.DOMParser()).parseFromString(xmlStr, "text/xml");

DMVAST.parser.load(vastXml, options, cb);
```

`2.x` - Fetch and parse a VAST or simply parse a VAST:
```Javascript
// To fetch and parse a VAST
const vastUrl = 'http://example.dailymotion.com/vast.xml';

// A Promise is returned
vastParser.getAndParseVAST(vastUrl, options);

// To parse an already fetched VAST xml
const vastXml = (new window.DOMParser()).parseFromString(xmlStr, "text/xml");

// A Promise is returned
vastParser.parseVAST(vastXml, options);
```

### Breaking changes in events

 * `resolved` becomes `VAST-resolved`
 * `resolving` becomes `VAST-resolving`

## Changes in Tracker API

### Constructor and properties
The constructor now accepts a `VASTClient` instance as first parameter if you need your istance of `VASTClient` to be updated by the `VASTTracker`.

If you don't need that, simply pass `null` as first parameter to the constructor.

### Breaking changes in methods signatures
These changes are mostly due to the semantic meaning of methods names.

 * `load()` becomes `trackImpression()`

`1.x` - Track an ad impression:
```Javascript
player.on('canplay', function() {
  vastTracker.load();
});
```

`2.x` - Track an ad impression:
```Javascript
player.on('canplay', () => {
  vastTracker.trackImpression();
});
```

### Breaking changes in events
No breaking changes in `VASTTracker` events.
