"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const reporter_1 = __importDefault(require("@wdio/reporter"));
const chalk_1 = __importDefault(require("chalk"));
class ConciseReporter extends reporter_1.default {
    constructor(options) {
        /**
        * make Concise reporter to write to output stream by default
        */
        super(Object.assign(options, { stdout: true }));
        // keep track of the order that suites were called
        this._suiteUids = [];
        this._suites = [];
        this._stateCounts = { failed: 0 };
    }
    onSuiteStart(suite) {
        this._suiteUids.push(suite.uid);
    }
    onSuiteEnd(suite) {
        this._suites.push(suite);
    }
    onTestFail() {
        this._stateCounts.failed++;
    }
    onRunnerEnd(runner) {
        this.printReport(runner);
    }
    /**
    * Print the Concise report to the screen
    * @param  {Object} runner Wdio runner
    */
    printReport(runner) {
        const header = chalk_1.default.yellow('========= Your concise report ==========');
        const output = [
            this.getEnviromentCombo(runner.capabilities),
            this.getCountDisplay(),
            ...this.getFailureDisplay()
        ];
        this.write(`${header}\n${output.join('\n')}\n`);
    }
    /**
    * Get the display for failing tests
    * @return {String} Count display
    */
    getCountDisplay() {
        const failedTestsCount = this._stateCounts.failed;
        return failedTestsCount > 0
            ? `Test${failedTestsCount > 1 ? 's' : ''} failed (${failedTestsCount}):`
            : 'All went well !!';
    }
    /**
    * Get display for failed tests, e.g. stack trace
    * @return {Array} Stack trace output
    */
    getFailureDisplay() {
        const output = [];
        this.getOrderedSuites().map(suite => suite.tests.map(test => {
            var _a;
            if (test.state === 'failed') {
                output.push(`  Fail : ${chalk_1.default.red(test.title)}`, 
                // @ts-ignore
                `    ${test.error.type} : ${chalk_1.default.yellow((_a = test.error) === null || _a === void 0 ? void 0 : _a.message)}`);
            }
        }));
        return output;
    }
    /**
     * Get suites in the order they were called
     * @return {Array} Ordered suites
     */
    getOrderedSuites() {
        const orderedSuites = [];
        this._suiteUids.map(uid => this._suites.map(suite => {
            if (suite.uid === uid) {
                orderedSuites.push(suite);
            }
        }));
        return orderedSuites;
    }
    /**
     * Get information about the enviroment
     * @param  {Object}  caps    Enviroment details
     * @param  {Boolean} verbose
     * @return {String}          Enviroment string
     */
    getEnviromentCombo(caps) {
        const device = caps.deviceName;
        const browser = caps.browserName || caps.browser;
        const version = caps.version || caps.platformVersion || caps.browser_version;
        const platform = caps.os ? (caps.os + ' ' + caps.os_version) : (caps.platform || caps.platformName);
        // mobile capabilities
        if (device) {
            const program = (caps.app || '').replace('sauce-storage:', '') || caps.browserName;
            const executing = program ? `executing ${program}` : '';
            return `${device} on ${platform} ${version} ${executing}`.trim();
        }
        return browser
            + (version ? ` (v${version})` : '')
            + (platform ? ` on ${platform}` : '');
    }
}
exports.default = ConciseReporter;
