"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const logger_1 = __importDefault(require("@wdio/logger"));
const got_1 = __importDefault(require("got"));
const util_1 = require("./util");
const log = (0, logger_1.default)('@wdio/browserstack-service');
class BrowserstackService {
    constructor(_options, _caps, _config) {
        this._options = _options;
        this._caps = _caps;
        this._config = _config;
        this._sessionBaseUrl = 'https://api.browserstack.com/automate/sessions';
        this._failReasons = [];
        this._scenariosThatRan = [];
        this._failureStatuses = ['failed', 'ambiguous', 'undefined', 'unknown'];
        // Cucumber specific
        const strict = Boolean(this._config.cucumberOpts && this._config.cucumberOpts.strict);
        // See https://github.com/cucumber/cucumber-js/blob/master/src/runtime/index.ts#L136
        if (strict) {
            this._failureStatuses.push('pending');
        }
    }
    _updateCaps(fn) {
        const multiRemoteCap = this._caps;
        if (multiRemoteCap.capabilities) {
            return Object.entries(multiRemoteCap).forEach(([, caps]) => fn(caps.capabilities));
        }
        return fn(this._caps);
    }
    /**
     * if no user and key is specified even though a browserstack service was
     * provided set user and key with values so that the session request
     * will fail
     */
    beforeSession(config) {
        if (!config.user) {
            config.user = 'NotSetUser';
        }
        if (!config.key) {
            config.key = 'NotSetKey';
        }
        this._config.user = config.user;
        this._config.key = config.key;
    }
    before(caps, specs, browser) {
        this._browser = browser;
        // Ensure capabilities are not null in case of multiremote
        if (this._isAppAutomate()) {
            this._sessionBaseUrl = 'https://api-cloud.browserstack.com/app-automate/sessions';
        }
        this._scenariosThatRan = [];
        return this._printSessionURL();
    }
    beforeSuite(suite) {
        this._fullTitle = suite.title;
    }
    beforeFeature(uri, feature) {
        this._fullTitle = feature.name;
        return this._updateJob({ name: this._fullTitle });
    }
    afterTest(test, context, results) {
        const { error, passed } = results;
        // Jasmine
        if (test.fullName) {
            const testSuiteName = test.fullName.slice(0, test.fullName.indexOf(test.description || '') - 1);
            if (this._fullTitle === 'Jasmine__TopLevel__Suite') {
                this._fullTitle = testSuiteName;
            }
            else if (this._fullTitle) {
                this._fullTitle = (0, util_1.getParentSuiteName)(this._fullTitle, testSuiteName);
            }
        }
        else {
            // Mocha
            this._fullTitle = `${test.parent} - ${test.title}`;
        }
        if (!passed) {
            this._failReasons.push((error && error.message) || 'Unknown Error');
        }
    }
    after(result) {
        // For Cucumber: Checks scenarios that ran (i.e. not skipped) on the session
        // Only 1 Scenario ran and option enabled => Redefine session name to Scenario's name
        if (this._options.preferScenarioName && this._scenariosThatRan.length === 1) {
            this._fullTitle = this._scenariosThatRan.pop();
        }
        const hasReasons = Boolean(this._failReasons.filter(Boolean).length);
        return this._updateJob({
            status: result === 0 ? 'passed' : 'failed',
            name: this._fullTitle,
            reason: hasReasons ? this._failReasons.join('\n') : undefined
        });
    }
    /**
     * For CucumberJS
     */
    afterScenario(world) {
        var _a;
        const status = (_a = world.result) === null || _a === void 0 ? void 0 : _a.status.toLowerCase();
        if (status === 'skipped') {
            this._scenariosThatRan.push(world.pickle.name || 'unknown pickle name');
        }
        if (status && this._failureStatuses.includes(status)) {
            const exception = ((world.result && world.result.message) ||
                (status === 'pending'
                    ? `Some steps/hooks are pending for scenario "${world.pickle.name}"`
                    : 'Unknown Error'));
            this._failReasons.push(exception);
        }
    }
    async onReload(oldSessionId, newSessionId) {
        if (!this._browser) {
            return Promise.resolve();
        }
        const hasReasons = Boolean(this._failReasons.filter(Boolean).length);
        let status = hasReasons ? 'failed' : 'passed';
        if (!this._browser.isMultiremote) {
            log.info(`Update (reloaded) job with sessionId ${oldSessionId}, ${status}`);
        }
        else {
            const browserName = this._browser.instances.filter((browserName) => this._browser && this._browser[browserName].sessionId === newSessionId)[0];
            log.info(`Update (reloaded) multiremote job for browser "${browserName}" and sessionId ${oldSessionId}, ${status}`);
        }
        await this._update(oldSessionId, {
            name: this._fullTitle,
            status,
            reason: hasReasons ? this._failReasons.join('\n') : undefined
        });
        this._scenariosThatRan = [];
        delete this._fullTitle;
        this._failReasons = [];
        await this._printSessionURL();
    }
    _isAppAutomate() {
        var _a, _b, _c;
        const browserDesiredCapabilities = ((_b = (_a = this._browser) === null || _a === void 0 ? void 0 : _a.capabilities) !== null && _b !== void 0 ? _b : {});
        const desiredCapabilities = ((_c = this._caps) !== null && _c !== void 0 ? _c : {});
        return !!browserDesiredCapabilities['appium:app'] || !!desiredCapabilities['appium:app'] || !!browserDesiredCapabilities.app || !!desiredCapabilities.app;
    }
    _updateJob(requestBody) {
        return this._multiRemoteAction((sessionId, browserName) => {
            log.info(browserName
                ? `Update multiremote job for browser "${browserName}" and sessionId ${sessionId}`
                : `Update job with sessionId ${sessionId}`);
            return this._update(sessionId, requestBody);
        });
    }
    _multiRemoteAction(action) {
        const { _browser } = this;
        if (!_browser) {
            return Promise.resolve();
        }
        if (!_browser.isMultiremote) {
            return action(_browser.sessionId);
        }
        return Promise.all(_browser.instances
            .filter(browserName => {
            const cap = (0, util_1.getBrowserCapabilities)(_browser, this._caps, browserName);
            return (0, util_1.isBrowserstackCapability)(cap);
        })
            .map((browserName) => (action(_browser[browserName].sessionId, browserName))));
    }
    _update(sessionId, requestBody) {
        const sessionUrl = `${this._sessionBaseUrl}/${sessionId}.json`;
        log.debug(`Updating Browserstack session at ${sessionUrl} with request body: `, requestBody);
        return got_1.default.put(sessionUrl, {
            json: requestBody,
            username: this._config.user,
            password: this._config.key
        });
    }
    async _printSessionURL() {
        if (!this._browser) {
            return Promise.resolve();
        }
        await this._multiRemoteAction(async (sessionId, browserName) => {
            const sessionUrl = `${this._sessionBaseUrl}/${sessionId}.json`;
            log.debug(`Requesting Browserstack session URL at ${sessionUrl}`);
            const response = await (0, got_1.default)(sessionUrl, {
                username: this._config.user,
                password: this._config.key,
                responseType: 'json'
            });
            if (!this._browser) {
                return;
            }
            const capabilities = (0, util_1.getBrowserCapabilities)(this._browser, this._caps, browserName);
            const browserString = (0, util_1.getBrowserDescription)(capabilities);
            log.info(`${browserString} session: ${response.body.automation_session.browser_url}`);
        });
    }
}
exports.default = BrowserstackService;
