import React, { useState, useRef, useEffect } from "react";
import "vimond-replay/index.css";

import dashjs from "dashjs";
import KeyHandler from "../KeyHandler";
import KeyCodes from "../KeyHandler/KeyCodes";

import { useStoreState } from "easy-peasy";

const Video = ({
  video,
  vodLivestream = false,
  livestream = false,
  onLoadEvent = () => {},
}) => {
  const videoRef = useRef(null);
  const [videoReload, setVideoReload] = useState(false);
  const [playing, setPlaying] = useState(true);
  const [totalTime, setTotalTime] = useState(0);
  const [currentTime, setCurrentTime] = useState({ timeToEnd: 0, time: 0 });
  const [showControls, setShowControls] = useState(true);

  const timer = useRef(null);
  const player = useRef(null);

  const controller = useStoreState((state) => state.controller);

  useEffect(() => {
    if (controller.data.categoryOpen !== null) player.current.setMute(true);
    else player.current?.setMute(false);
  }, [controller.data.categoryOpen]);

  useEffect(() => {
    setVideoReload(true);
    setShowControls(false);
    setTimeout(() => {
      setVideoReload(false);
    }, 200);
  }, [video]);

  useEffect(() => {
    if (!videoReload) {
      try {
        document.getElementById("video").release();
      } catch (e) {
        console.log("Ref bind to channel failed", e);
      }
      _initPlayer();
    }
  }, [videoReload]);

  const onStartEvent = () => {
    // console.log("Time", player.current.duration())
    setTotalTime(player.current.duration());
    onLoadEvent();
  };

  const _onProgress = (e) => {
    // console.log("PROGRESS", e)
    setCurrentTime(e);
    // setCurrentTime()
  };

  const _initPlayer = () => {
    if (player.current !== null) {
      player.current.destroy();
      player.current.off(
        dashjs.MediaPlayer.events["PLAYBACK_STARTED"],
        onStartEvent
      );
      player.current.off(
        dashjs.MediaPlayer.events["STREAM_INITIALIZED"],
        onStartEvent
      );
      player.current.off(
        dashjs.MediaPlayer.events["PLAYBACK_TIME_UPDATED"],
        _onProgress
      );
    }
    try {
      document.getElementById("video").bindToCurrentChannel();
    } catch (e) {
      console.log("Ref bind to channel failed", e);
    }
    player.current = dashjs.MediaPlayer().create();
    player.current.initialize(videoRef.current, video, true);
    setPlaying(true);

    // if (player.current.isPaused()) {
    //     player.current.play()
    // }

    player.current.on(
      dashjs.MediaPlayer.events["PLAYBACK_STARTED"],
      onStartEvent
    );
    player.current.on(
      dashjs.MediaPlayer.events["STREAM_INITIALIZED"],
      onStartEvent
    );

    if (!livestream && !vodLivestream) {
      player.current.on(
        dashjs.MediaPlayer.events["PLAYBACK_TIME_UPDATED"],
        _onProgress
      );
      if (!showControls) {
        setShowControls(true);
        startHide();
      }
    }
  };

  const startHide = () => {
    if (timer.current !== null) {
      clearTimeout(timer.current);
      timer.current = null;
    }
    timer.current = setTimeout(() => {
      setShowControls(false);
      timer.current = null;
    }, 4000);
  };

  const doForward = () => {
    setShowControls(true);
    startHide();
    if (!livestream && currentTime.time + 10 < totalTime) {
      player.current.seek(currentTime.time + 10);
      setCurrentTime((ct) =>
        Object.assign({}, ct, {
          time: ct.time + 10,
          timeToEnd: ct.timeToEnd - 10,
        })
      );
    } else {
      player.current.seek(totalTime);
      setCurrentTime((ct) =>
        Object.assign({}, ct, { time: totalTime, timeToEnd: 0 })
      );
    }
  };

  const doRewind = () => {
    setShowControls(true);
    startHide();
    if (!livestream && currentTime.time - 10 > 0) {
      player.current.seek(currentTime.time - 10);
      setCurrentTime((ct) =>
        Object.assign({}, ct, {
          time: ct.time - 10,
          timeToEnd: ct.timeToEnd + 10,
        })
      );
    } else {
      player.current.seek(0);
      setCurrentTime((ct) =>
        Object.assign({}, ct, { time: 0, timeToEnd: totalTime })
      );
    }
  };

  const togglePlay = () => {
    if (player.current.isPaused()) {
      player.current.play();
      setPlaying(true);
    } else {
      player.current.pause();
      setPlaying(false);
    }
  };

  const onKeyDown = (keyCode) => {
    if (!livestream && !vodLivestream) {
      if ([KeyCodes.VK_PLAY_PAUSE, KeyCodes.VK_SPACE].includes(keyCode)) {
        togglePlay();
      } else if ([KeyCodes.VK_PLAY].includes(keyCode)) {
        if (player.current.isPaused()) {
          player.current.play();
          setPlaying(true);
        }
      } else if ([KeyCodes.VK_PAUSE].includes(keyCode)) {
        if (!player.current.isPaused()) {
          player.current.pause();
          setPlaying(false);
        }
      } else if ([KeyCodes.VK_STOP, KeyCodes.VK_DEBUG_S].includes(keyCode)) {
        if (currentTime.time > 0) {
          setShowControls(true);
          startHide();
          setPlaying(false);
          player.current.pause();
          player.current.seek(0);
        }
      } else if (
        [KeyCodes.VK_DEBUG_F, KeyCodes.VK_FAST_FWD].includes(keyCode)
      ) {
        doForward();
      } else if ([KeyCodes.VK_DEBUG_D, KeyCodes.VK_REWIND].includes(keyCode)) {
        doRewind();
      } else if (
        !controller.data.isMenuOpen &&
        [KeyCodes.VK_RIGHT].includes(keyCode)
      ) {
        doForward();
      } else if (
        !controller.data.isMenuOpen &&
        [KeyCodes.VK_LEFT].includes(keyCode)
      ) {
        doRewind();
      }
      if (
        !controller.data.isMenuOpen &&
        [KeyCodes.VK_ENTER].includes(keyCode)
      ) {
        togglePlay();
      }
    }
  };

  if (videoReload) {
    return (
      <div style={{ height: "100vh", width: "100vw", background: "#000000" }} />
    );
  } else {
    return (
      <>
      {/* <div className={"image-advertisment-overlay"}><image src="/img/baeckerei_ek.png"/></div> */}
       <video id={"image-advertisment"} className={"image-advertisment-overlay"}
          ref={videoRef}
          //height={720}
          style={{ background: "#000000" }}
          // width={1280}
        >
          <source src={video} />
          Your browser does not support the video tag.
        </video>
        {!playing && (
          <div className={"video-indicator-overlay"}>
            <img src={"img/play-button-2.png"} alt={"play"} />
          </div>
        )}
        {!livestream && !vodLivestream && showControls && (
          <div className={"video-controls"}>
            <p>
              {new Date(currentTime.time * 1000).toISOString().substr(11, 8)}
            </p>
            <div className={"progress"}>
              <div
                className={"bar"}
                style={{
                  width: (currentTime.time / totalTime) * 100 + "%",
                  background: window.premio_config.background[0],
                }}
              />
            </div>
            <p>
              {new Date(currentTime.timeToEnd * 1000)
                .toISOString()
                .substr(11, 8)}
            </p>
          </div>
        )}
        <KeyHandler onKeyDown={onKeyDown} />
      </>
    );
  }
};

export default Video;
