<?php

/**
 * apDeliveryLogRedis for Revive Adserver and OpenX Source
 *
 * @author Matteo Beccati
 * @copyright 2013-2015 AdserverPlugins.com - All rights reserved
 *
 */


require_once MAX_PATH.'/lib/pear/Date.php';
require_once MAX_PATH.'/lib/OA/Cache.php';
require_once MAX_PATH.'/lib/OX/Plugin/Component.php';

require_once __DIR__.'/Admin.php';


class AP_DeliveryLogRedis_DAL_Distributed extends AP_DeliveryLogRedis_DAL_Admin
{
    protected $oRedisDest;

    public function __construct($oDbh, $oRedisDest)
    {
        parent::__construct($oDbh);

        $this->checkVersion($oRedisDest);

        $this->oRedisDest = $oRedisDest;
    }

    protected function checkVersion($oRedis, $type = 'remote')
    {
        if ($oRedis->getType() == AP_Redis::TYPE_PHP) {
            throw new Exception("Please install the php redis extension");
        }

        try {
            $info = $oRedis->info('server');
        } catch (RedisException $e) {
            $info = $oRedis->info();
        }

        if (empty($info['redis_version'])) {
            throw new Exception("Cannot get the {$type} Redis version");
        }

        if (version_compare($info['redis_version'], '2.6.0', '<')) {
            throw new Exception(ucfirst($type)." Redis version {$info['redis_version']} is < 2.6.");
        }
    }

    public function migrate($oRedis, $oNow = null)
    {
        $this->checkVersion($oRedis, 'local');

        return parent::migrate($oRedis, $oNow);
    }

    protected function migrateComponent(AP_DeliveryLogRedis_MaintenanceTask $task, $oRedis, $oNow)
    {
        if ($task instanceof AP_DeliveryLogRedis_EnhancedMaintenanceTask) {
            $task->migrateToRedis($oRedis, $this->oRedisDest, $oNow);
        }
    }

    protected function migrateAndEval($oRedis, $key, $lua)
    {
        $src = $oRedis->multi()
            ->dump($key)
            ->del($key)
            ->exec();

        if (!empty($src[0])) {
            $tmp = $key.'_'.md5(uniqid('', true));

            if ($this->oRedisDest->restore($tmp, 0, $src[0])) {
                if ($this->oRedisDest->eval($lua, array($tmp, $key), 2)) {
                    $this->oRedisDest->del($tmp);
                    return true;
                }
            }

            // Something went wrong
            $oRedis->restore($tmp, 0, $src[0]);
        }

    }

    protected function migrateBucketKey($oRedis, $type, $key, $t_stamp)
    {
        return $this->migrateAndEval($oRedis, $key, "
            local res = redis.call('hgetall', KEYS[1])
            for i = 1, #res, 2 do
                redis.call('hincrby', KEYS[2], res[i], res[i+1])
            end
            return redis.status_reply('OK')
        ");
    }

    protected function migrateRawBucket($oRedis, $bucket, $table)
    {
        $key = $oRedis->getPrefix().$bucket;

        return $this->migrateAndEval($oRedis, $key, "
            local l = redis.call('llen', KEYS[1])
            for i = 1, l do
                local r = redis.call('lpop', KEYS[1])
                redis.call('rpush', KEYS[2], r)
            end
            return redis.status_reply('OK')
        ");
    }
}
