<?php

/*
 * @copyright   2016 Mautic, Inc. All rights reserved
 * @author      Mautic, Inc
 *
 * @link        https://mautic.org
 *
 * @license     GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace MauticPlugin\MauticMyupcharSmsBundle\Controller;

use Mautic\CoreBundle\Controller\FormController;
use Mautic\CoreBundle\Helper\Chart\LineChart;
use Mautic\LeadBundle\Controller\EntityContactsTrait;
use MauticPlugin\MauticMyupcharSmsBundle\Entity\Myupchar;

/**
 * Class MonitoringController.
 */
class MyupcharController extends FormController
{

    public function indexAction($page = 1)
    {
         //set some permissions
         $permissions = $this->get('mautic.security')->isGranted(
            [
                'myupchar:myupchars:viewown',
                'myupchar:myupchars:viewother',
               
            ],
            'RETURN_ARRAY'
        );

        if (!$permissions['myupchar:myupchars:viewown'] && !$permissions['myupchar:myupchars:viewother']) {
            return $this->accessDenied();
        }
        $session = $this->get('session');

        $this->setListFilters();

        
        $model = $this->getModel('myupchar');

        //set limits
        $limit = $session->get('mautic.myupchar.monitoring.limit', $this->container->getParameter('mautic.default_pagelimit'));
        $start = ($page === 1) ? 0 : (($page - 1) * $limit);
        if ($start < 0) {
            $start = 0;
        }

        $search = $this->request->get('search', $session->get('mautic.myupchar.filter', ''));
        $session->set('mautic.myupchar.filter', $search);

        $filter = ['string' => $search, 'force' => []];

        $orderBy    = $session->get('mautic.myupchar.orderby', 'e.message');
        $orderByDir = $session->get('mautic.myupchar.orderbydir', 'DESC');

        $monitoringList = $model->getEntities(
            [
                'start'      => $start,
                'limit'      => $limit,
                'filter'     => $filter,
                'orderBy'    => $orderBy,
                'orderByDir' => $orderByDir,
            ]
        );

        $count = count($monitoringList);
        
        if ($count && $count < ($start + 1)) {
            //the number of entities are now less then the current asset so redirect to the last asset
            if ($count === 1) {
                $lastPage = 1;
            } else {
                $lastPage = (floor($limit / $count)) ?: 1;
            }
            $session->set('mautic.myupchar.monitoring.page', $lastPage);
            $returnUrl = $this->generateUrl('mautic_myupchar_index', ['page' => $lastPage]);

            return $this->postActionRedirect(
                [
                    'returnUrl'       => $returnUrl,
                    'viewParameters'  => ['page' => $lastPage],
                    'contentTemplate' => 'MauticMyupcharSmsBundle:Monitoring:index',
                    'passthroughVars' => [
                        'activeLink'    => '#mautic_myupchar_index',
                        'mauticContent' => 'monitoring',
                    ],
                ]
            );
        }

        //set what asset currently on so that we can return here after form submission/cancellation
        $session->set('mautic.myupchar.monitoring.page', $page);

        $tmpl = $this->request->isXmlHttpRequest() ? $this->request->get('tmpl', 'index') : 'index';

        return $this->delegateView(
            [
                'viewParameters' => [
                    'searchValue' => $search,
                    'items'       => $monitoringList,
                    'limit'       => $limit,
                    'model'       => $model,
                    'tmpl'        => $tmpl,
                    'page'        => $page,
                ],
                'contentTemplate' => 'MauticMyupcharSmsBundle:Myupchar:list.html.php',
                'passthroughVars' => [
                    'activeLink'    => '#mautic_myupchar_index',
                    'mauticContent' => 'monitoring',
                    'route'         => $this->generateUrl('mautic_myupchar_index', ['page' => $page]), 
                ],
            ]
        );
    }

    public function newAction()
    {
        if (!$this->container->get('mautic.security')->isGranted('plugin:mauticMyupcharSms:myupchar:create')) {
            return $this->accessDenied();
        }

        $action = $this->generateUrl('mautic_myupchar_action', ['objectAction' => 'new']);

   
        $model = $this->getModel('myupchar');

        $entity  = $model->getEntity();
        $method  = $this->request->getMethod();
        $session = $this->get('session');

        // build the form
        $form = $model->createForm(
            $entity,
            $this->get('form.factory'),
            $action
        );

        // Set the page we came from
        $page = $session->get('mautic.myupchar.page', 1);
        ///Check for a submitted form and process it
        if ($method == 'POST') {
            $viewParameters = ['page' => $page];
            $template       = 'MauticMyupcharSmsBundle:Myupchar:index';

            $valid = false;
            if (!$cancelled = $this->isFormCancelled($form)) {
                if ($valid = $this->isFormValid($form)) {
                    
                    //form is valid so process the data
                    $model->saveEntity($entity);
                   
                    // update the audit log
                    $this->updateAuditLog($entity, 'create');

                    $this->addFlash(
                        'mautic.core.notice.created',
                        [
                            '%message%'      => $entity->getMessage(),
                            '%menu_link%' => 'mautic_myupchar_index',
                            '%url%'       => $this->generateUrl(
                                'mautic_myupchar_action',
                                [
                                    'objectAction' => 'edit',
                                    'objectId'     => $entity->getId(),
                                ]
                            ),
                        ]
                    );
                    
                    if (!$form->get('buttons')->get('save')->isClicked()) {
                        //return edit view so that all the session stuff is loaded
                        return $this->editAction($entity->getId(), true);
                    }

                    $viewParameters = [
                        // 'objectAction' => 'index',
                    ];
                    $template = 'MauticMyupcharSmsBundle:Myupchar:index';
                }
            }
            $returnUrl = $this->generateUrl('mautic_myupchar_index', $viewParameters);

            if ($cancelled || ($valid && $form->get('buttons')->get('save')->isClicked())) {
                return $this->postActionRedirect(
                    [
                        'returnUrl'       => $returnUrl,
                        'viewParameters'  => $viewParameters,
                        'contentTemplate' => $template,
                        'passthroughVars' => [
                            'activeLink'    => 'mautic_myupchar_index',
                            'mauticContent' => 'monitoring',
                        ],
                    ]
                );
            }
        }

        return $this->delegateView(
            [
                'viewParameters' => [
                    'tmpl'   => $this->request->isXmlHttpRequest() ? $this->request->get('tmpl', 'index') : 'index',
                    'entity' => $entity,
                    'form'   => $form->createView(),
                ],
                'contentTemplate' => 'MauticMyupcharSmsBundle:Myupchar:form.html.php',
                'passthroughVars' => [
                    'activeLink'    => '#mautic_myupchar_index',
                    'mauticContent' => 'monitoring',
                    'route'         => $this->generateUrl(
                        'mautic_myupchar_action',
                        [
                            'objectAction' => 'new',
                            'objectId'     => $entity->getId(),
                        ]
                    ),
                ],
            ]
        );
    }

    public function editAction($objectId)
    {
        if (!$this->container->get('mautic.security')->isGranted('plugin:mauticMyupcharSms:myupchar:edit')) {
            return $this->accessDenied();
        }

        $action = $this->generateUrl('mautic_myupchar_action', ['objectAction' => 'edit', 'objectId' => $objectId]);


        $model = $this->getModel('myupchar');

        $entity  = $model->getEntity($objectId);
        $session = $this->container->get('session');

        // Set the page we came from
        $page = $session->get('mautic.myupchar.page', 1);

        //set the return URL
        $returnUrl = $this->generateUrl('mautic_myupchar_index', ['page' => $page]);

        $postActionVars = [
            'returnUrl'       => $returnUrl,
            'viewParameters'  => ['page' => $page],
            'contentTemplate' => 'MauticMyupcharSms:Myupchar:index',
            'passthroughVars' => [
                'activeLink'    => 'mautic_myupchar_index',
                'mauticContent' => 'monitoring',
            ],
        ];

        //not found
        if ($entity === null) {
            return $this->postActionRedirect(
                array_merge(
                    $postActionVars,
                    [
                        'flashes' => [
                            [
                                'type'    => 'error',
                                'msg'     => 'mautic.myupchar.error.notfound',
                                'msgVars' => ['%id%' => $objectId],
                            ],
                        ],
                    ]
                )
            );
        }

        // build the form
        $form = $model->createForm(
            $entity,
            $this->get('form.factory'),
            $action
        );

        ///Check for a submitted form and process it
        if ($this->request->getMethod() == 'POST') {
            $valid = false;
            if (!$cancelled = $this->isFormCancelled($form)) {
                if ($valid = $this->isFormValid($form)) {
                    //form is valid so process the data
                    $model->saveEntity($entity, $form->get('buttons')->get('save')->isClicked());

                    // update the audit log
                    $this->updateAuditLog($entity, 'update');

                    $this->addFlash(
                        'mautic.core.notice.updated',
                        [
                            '%message%'      => $entity->getMessage(),
                            '%menu_link%' => 'mautic_myupchar_index',
                            '%url%'       => $this->generateUrl(
                                'mautic_myupchar_action',
                                [
                                    'objectAction' => 'edit',
                                    'objectId'     => $entity->getId(),
                                ]
                            ),
                        ],
                        'warning'
                    );
                }
            } else {
                $model->unlockEntity($entity);
            }

            if ($cancelled || ($valid && $form->get('buttons')->get('save')->isClicked())) {
                $viewParameters = [
                    // 'objectAction' => 'index',
                    // 'objectId'     => $entity->getId(),
                ];

                return $this->postActionRedirect(
                    array_merge(
                        $postActionVars,
                        [
                            'returnUrl'       => $this->generateUrl('mautic_myupchar_index', $viewParameters),
                            'viewParameters'  => $viewParameters,
                            'contentTemplate' => 'MauticMyupcharSmsBundle:Myupchar:index',
                        ]
                    )
                );
            }
        } else {
            //lock the entity
            $model->lockEntity($entity);
        }

        return $this->delegateView(
            [
                'viewParameters' => [
                    'tmpl'   => $this->request->isXmlHttpRequest() ? $this->request->get('tmpl', 'index') : 'index',
                    'entity' => $entity,
                    'form'   => $form->createView(),
                ],
                'contentTemplate' => 'MauticMyupcharSmsBundle:Myupchar:form.html.php',
                'passthroughVars' => [
                    'activeLink'    => '#mautic_myupchar_index',
                    'mauticContent' => 'monitoring',
                    'route'         => $this->generateUrl(
                        'mautic_myupchar_action',
                        [
                            'objectAction' => 'edit',
                            'objectId'     => $entity->getId(),
                        ]
                    ),
                ],
            ]
        );
    }

   

    public function deleteAction($objectId)
    {
        if (!$this->get('mautic.security')->isGranted('plugin:mauticMyupcharSms:myupchar:delete')) {
            return $this->accessDenied();
        }

        $session   = $this->get('session');
        $page      = $session->get('mautic.myupchar.page', 1);
        $returnUrl = $this->generateUrl('mautic_myupchar_index', ['page' => $page]);
        $flashes   = [];

        $postActionVars = [
            'returnUrl'       => $returnUrl,
            'viewParameters'  => ['page' => $page],
            'contentTemplate' => 'MauticMyupcharSmsBundle:Myupchar:index',
            'passthroughVars' => [
                'activeLink'    => 'mautic_myupchar_index',
                'mauticContent' => 'monitoring',
            ],
        ];

        if ($this->request->getMethod() == 'POST') {
            
    
            $model  = $this->getModel('myupchar');
            $entity = $model->getEntity($objectId);

            if ($entity === null) {
                $flashes[] = [
                    'type'    => 'error',
                    'msg'     => 'mautic.myupchar.error.notfound',
                    'msgVars' => ['%id%' => $objectId],
                ];
            } elseif ($model->isLocked($entity)) {
                return $this->isLocked($postActionVars, $entity, 'plugin.mauticMyupcharSms.myupchar');
            }

            // update the audit log
            $this->updateAuditLog($entity, 'delete');

            // then delete the record
            $model->deleteEntity($entity);

            $flashes[] = [
                'type'    => 'notice',
                'msg'     => 'mautic.core.notice.deleted',
                'msgVars' => [
                    '%message%' => $entity->getMessage(),
                    '%id%'   => $objectId,
                ],
            ];
        } //else don't do anything

        return $this->postActionRedirect(
            array_merge(
                $postActionVars,
                [
                    'flashes' => $flashes,
                ]
            )
        );
    }

    public function batchDeleteAction()
    {
       
        if (!$this->container->get('mautic.security')->isGranted('plugin:mauticMyupcharSms:myupchar:delete')) {
            return $this->accessDenied();
        }

        $session   = $this->get('session');
        $page      = $session->get('mautic.myupchar.page', 1);
        $returnUrl = $this->generateUrl('mautic_myupchar_index', ['page' => $page]);
        $flashes   = [];

        $postActionVars = [
            'returnUrl'       => $returnUrl,
            'viewParameters'  => ['page' => $page],
            'contentTemplate' => 'MauticMyupcharSmsBundle:Myupchar:index',
            'passthroughVars' => [
                'activeLink'    => '#mautic_myupchar_index',
                'mauticContent' => 'monitoring',
            ],
        ];

        if ($this->request->getMethod() == 'POST') {

            $model = $this->getModel('myupchar');

            $ids       = json_decode($this->request->query->get('ids', ''));
            $deleteIds = [];

            // Loop over the IDs to perform access checks pre-delete
            foreach ($ids as $objectId) {
                $entity = $model->getEntity($objectId);

                if ($entity === null) {
                    $flashes[] = [
                        'type'    => 'error',
                        'msg'     => 'mautic.myupchar.error.notfound',
                        'msgVars' => ['%id%' => $objectId],
                    ];
                } elseif ($model->isLocked($entity)) {
                    $flashes[] = $this->isLocked($postActionVars, $entity, 'myupchar', true);
                } else {
                    $deleteIds[] = $objectId;
                }
            }

            // Delete everything we are able to
            if (!empty($deleteIds)) {
                $entities = $model->deleteEntities($deleteIds);

                $flashes[] = [
                    'type'    => 'notice',
                    'msg'     => 'mautic.myupchar.notice.batch_deleted',
                    'msgVars' => [
                        '%count%' => count($entities),
                    ],
                ];
            }
        } //else don't do anything

        return $this->postActionRedirect(
            array_merge(
                $postActionVars,
                [
                    'flashes' => $flashes,
                ]
            )
        );
    }
        /*
     * Update the audit log
     */
    public function updateAuditLog(Myupchar $myupchar, $action)
    {
        $log = [
            'bundle'    => 'plugin.mauticMyupcharSms',
            'object'    => 'myupchar',
            'objectId'  => $myupchar->getId(),
            'action'    => $action,
            'details'   => ['message' => $myupchar->getMessage()],
            'ipAddress' => $this->container->get('mautic.helper.ip_lookup')->getIpAddressFromRequest(),
        ];

        $this->getModel('core.auditLog')->writeToLog($log);
    }

  
}
