package redisclient

import (
	"alpha/addelivery/structure"
	config "alpha/configuration"
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"github.com/go-redis/redis"
)

var cxt = context.Background()

var (
	client = &RedisClient{}
)

type RedisClient struct {
	c *redis.Client
}

// GetClient get the redis client
func Initialize() *RedisClient {

	host := config.GetConfig("host", "redis")
	port := config.GetConfig("port", "redis")
	pass := config.GetConfig("password", "redis")

	c := redis.NewClient(&redis.Options{
		Addr:     host + ":" + port,
		Password: pass,
		DB:       0,
	})

	if err := c.Ping(cxt).Err(); err != nil {
		fmt.Println("Unable to connect to redis " + err.Error())
	}
	client.c = c
	return client
}

func (client *RedisClient) SetKey(key string, value interface{}, expire int) error {

	var expiration time.Duration
	switch {
	case expire == 0:
		expiration = time.Minute * 0
	case expire == 1:
		expiration = time.Second * 300
	}

	err := client.c.Set(cxt, key, value, expiration).Err()
	if err != nil {
		return err
	}
	return nil
}

func (client *RedisClient) GetKey(key string, src interface{}) error {
	val, err := client.c.Get(cxt, key).Result()
	if err == redis.Nil || err != nil {
		return err
	}
	err = json.Unmarshal([]byte(val), &src)
	if err != nil {
		return err
	}
	return nil
}
func (client *RedisClient) CappingGetKey(key string) (string, error) {
	src, err := client.c.Get(cxt, key).Result()
	if err == redis.Nil || err != nil {
		return src, err
	}

	return src, nil
}
func (client *RedisClient) HSetKey(key string, value map[string]interface{}) error {

	err := client.c.HMSet(cxt, key, value)
	if err != nil {
		fmt.Println(err)
	}
	return nil
}

func (client *RedisClient) HGetKey(key string, src map[string]string) error {

	val, err := client.c.HGetAll(cxt, key).Result()
	if err == redis.Nil || err != nil {
		return err
	}
	jsondata, err := json.Marshal(val)
	if err != nil {
		fmt.Println(err)
	}
	err = json.Unmarshal(jsondata, &src)
	if err != nil {
		return err
	}
	return nil
}

func (client *RedisClient) HFGetKey(key string, bannersize *structure.BannerSize) error {
	val, err := client.c.HGet(cxt, "Admaru_Banner_Size", key).Result()

	if err == redis.Nil || err != nil {
		return err
	}
	b := strings.Split(val, "X")
	bannersize.Width = b[0]
	bannersize.Height = b[1]

	return nil
}

func (client *RedisClient) OSHFGetKey(key string, os_val *structure.OSValue) error {
	val, err := client.c.HGet(cxt, "Target_OS", key).Result()
	os_val.OS_val = val

	if err == redis.Nil || err != nil {
		return err
	}

	return nil
}

func (client *RedisClient) GetKeyByField(key string, keyname string, Provider map[string]string) error {
	val, err := client.c.HGet(cxt, keyname, key).Result()

	err = json.Unmarshal([]byte(val), &Provider)

	if err != nil {
		return err
	}
	if err == redis.Nil || err != nil {
		return err
	}

	return nil
}

// Exists returns true if key exists.
func (client *RedisClient) IfExists(key string) int64 {
	rp, err := client.c.Exists(cxt, key).Result()
	if err != nil {
		fmt.Println(err)
	}
	return rp
}
